/**
 * 
 */
package gov.va.med.mhv.phr.backing;

import gov.va.med.mhv.core.util.MessagesStringBuilder;
import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.foiapareporting.transfer.PhrRequest;
import gov.va.med.mhv.phr.util.FoiaUtil;
import gov.va.med.mhv.phr.util.SessionManagement;
import gov.va.med.mhv.usermgmt.enumeration.ExtractEnablementStatus;
import gov.va.med.mhv.usermgmt.enumeration.ExtractType;
import gov.va.med.mhv.usermgmt.service.PatientServiceResponse;
import gov.va.med.mhv.usermgmt.service.delegate.ExtractServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.InPersonAuthenticationServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.util.ExtractEnablementStatusUtils;

import java.io.IOException;
import java.security.Principal;
import java.util.Collection;
import java.util.Date;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.messages.Message;
import org.tigris.atlas.messages.Messages;
import org.tigris.atlas.service.ServiceResponse;
import org.tigris.atlas.service.StringServiceResponse;

import com.bea.netuix.servlets.controls.content.backing.AbstractJspBacking;
import com.bea.netuix.servlets.controls.portlet.backing.PortletBackingContext;
import com.bea.portlet.PostbackURL;

/**
 * @author Rob Proper (Aquilent Inc.)
 *
 */
public abstract class ExtractBaseBacking extends AbstractJspBacking {
    
    private static final Log LOG = LogFactory.getLog(
        ExtractBaseBacking.class);

    /**
     * 
     */
    private static final long serialVersionUID = 4228981738120866920L;

    private static final InPersonAuthenticationServiceDelegate IPA_DELEGATE = 
        ServiceDelegateFactory.createInPersonAuthenticationServiceDelegate();

    private static final ExtractServiceDelegate EXTRACT_DELEGATE = 
        ServiceDelegateFactory.createExtractServiceDelegate();
    
    protected boolean allowAdvancedPatient(Patient patient, HttpServletRequest request) {
    	return false;
    }

    /*
     * (non-Javadoc)
     * @see com.bea.netuix.servlets.controls.content.backing.JspBacking#
     * preRender(javax.servlet.http.HttpServletRequest, 
     * javax.servlet.http.HttpServletResponse)
     */
    public boolean preRender(HttpServletRequest request, 
        HttpServletResponse response) 
    {
        boolean redirectToPortletBegin = false;
System.out.println("allowAdvancedPatient(getPatient(request), request)="+allowAdvancedPatient(getPatient(request), request));        
        if (!isUserAuthenticatedPatient(request)&&!allowAdvancedPatient(getPatient(request), request)) {
            LOG.debug("User is not an IPAed patient");
            redirectToPortletBegin = true;
        } else {
            ExtractEnablementStatus status = getEnablementStatus();
            setExtractEnablementStatus(request, status);

            if (isExtractUpdateable(status)) {
                updateIfNecessary(request);
            }
            
            boolean isViewable = isExtractViewable(status);
            if (isViewable) {
                recordToFoiaIfNeeded(request);
            } 
            redirectToPortletBegin = !isViewable;
        }
        if (redirectToPortletBegin && !isPortletBeginAction(request)) {
            // Only redirect if not already a request for begin action,
            // such that not attempting to continously redirect to the
            // begin action
            sendRedirect(response, createPortletBeginUrl(request, 
                response));
            return false;
        }
        return true;
    }

    private void recordToFoiaIfNeeded(HttpServletRequest request) {
        if (needToRecordFoia(request)) {
            LOG.debug("Recording a FOIA request");
            Patient user = SessionManagement.getInstance().getPatient(
                request);
            Collection<PhrRequest> foiaRequests = FoiaUtil.
                recordFoiaRequestInitiated(user);
            FoiaUtil.recordFoiaRequestsGranted(foiaRequests);
            SessionManagement.getInstance().setLastFoiaCountTime(request);
        }
    }
    
    private boolean needToRecordFoia(HttpServletRequest request) {
    	if ( FoiaUtil.isFoiaCountable(getExtractType()) ) {
    		Date lastUpdate = SessionManagement.getInstance().getLastFoiaCountTime(request);
    		return lastUpdate == null || !DateUtils.isSameDay(lastUpdate, new Date());
    	}
        return false;
	}

	/**
     *Method used to query the Extract service for an enablement status
     *@return ExtractEnablement enum type.
     */
    private ExtractEnablementStatus getEnablementStatus(){
    	 StringServiceResponse response = getExtractDelegate().
             getExtractViewStatus(getExtractType());
    	 ExtractEnablementStatus status = ExtractEnablementStatusUtils.
             getEnablementEnumeration(response.getString());
    	 if (LOG.isDebugEnabled()) {
    		 LOG.debug("Extract enablement status is " + status);
    	 }
    	 return status;
    }
    
    /**
     * Determine if the user in the session is an authenticated patient and
     * set the patient in the session.
     * @param request The request to obtain the session from
     * @return True if the user in the session is an authenticated patient;
     * false otherwise.
     */
    private boolean isUserAuthenticatedPatient(HttpServletRequest request) {
//        Patient patient = getPatient(request); 
//        if (patient == null) {
    	  Patient patient = getIPAedPatient(request).getPatient();
    	  if(patient != null) {
            setPatient(request, patient);
    	  }
//        }
        return patient != null;
    }
    
    /**
     * Get the patient if IPAed for the user principal in the session
     * @param request The request to obtain the session from
     * @return The patient service response containing the patient if IPAed,
     * or null if not IPAed. 
     */
    private PatientServiceResponse getIPAedPatient(
        HttpServletRequest request) 
    {
        Precondition.assertNotNull("request", request);
        Principal principal = request.getUserPrincipal();
        Precondition.assertNotNull("principal", principal);
        Precondition.assertNotBlank("principal.name", principal.getName());
        PatientServiceResponse serviceResponse = getIPADelegate().
            getIPAedPatientForUser(principal.getName());
        if ( ! hasNotAPatientMessage(serviceResponse) ) {
        	yieldOnError(serviceResponse, "Failed to obtain IPAed patient");
        }
        return serviceResponse;
    }
    
    private boolean hasNotAPatientMessage(ServiceResponse serviceResponse) {
		Messages msgs = serviceResponse.getMessages();
		for(Object errObj : msgs.getErrorMessages()) {
			Message m = (Message)errObj;
			if ( "user.not.a.patient".equals(m.getKey()) ) return true;
		}
		return false;
	}

	/**
     * Requests an update of the extract, if it has not already been requested
     * in the session.
     * @param request The request to obtain the session from.
     */
    private void updateIfNecessary(HttpServletRequest request) {
        if (!isExtractUpdated(request)) {
            // Request a extract update if the extract has not been updated
            updateExtract(request);
            setExtractUpdated(request, true);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Extract " + getExtractType().getName() 
                + " has already been updated for patient "
                + getUserName(request));
        }
    }

    /**
     * Determine if the request is for the begin action
     * @param request The request to obtain the session from
     * @return True if the extract is disabled and the action is not a begin 
     * action; false otherwise.
     */
    @SuppressWarnings("deprecation")
    private boolean isPortletBeginAction(HttpServletRequest request) {
        String action = getPortletActionName(request);
        return (action == null) 
            || "begin".equalsIgnoreCase(action)
            || "begin.do".equalsIgnoreCase(action);
    }
    
    /**
     * Get the simple name of the portlet action from the request.
     * The simple name does not include the controller's path.
     * @param request The request to obtain the action from.
     * @return The name of the action. null if not specified.
     */
    private String getPortletActionName(HttpServletRequest request) {
        return StringUtils.substringAfterLast(getPortletActionFullName(request), 
            "/");
    }
    
    /**
     * Get the name of the portlet action from the request
     * @param request The request to obtain the action from.
     * @return The name of the action. null if not specified.
     */
    private String getPortletActionFullName(HttpServletRequest request) {
        return request.getParameter(createActionParameter(request, 
            "actionOverride"));
    }

    private String createPortletActionValue(HttpServletRequest request, 
        String actionName) 
    {
        return "/" + getControllerClass().getPackage().getName().replace('.', 
            '/') + "/" + actionName;
    }

    private String createActionParameter(HttpServletRequest request, 
        String parameterName) 
    {
        return createWindowLabel(request) + "_" + parameterName;
    }

    private String createWindowLabel(HttpServletRequest request) {
        return PortletBackingContext.getPortletBackingContext(request).
                getPortletUri().replace('/', '_').replace('.', '_');
    }

    protected void sendRedirect(HttpServletResponse response, String url) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Redirecting to [" + url + "]");
        }
        try {
            if (!response.isCommitted()) {
                response.sendRedirect(url);
            } else {
                LOG.error("Failed to redirect to [" + url 
                    + "]. Response already committed");
            }
        } catch (IOException e) {
            throw new RuntimeException("Unable to redirect to [" + url + "]", 
                e);
        }
    }
    
    /**
     * Denotes the extract type supported by this extract busines service
     * implementation. 
     * @return The extract type.
     */
    protected abstract ExtractType getExtractType();

    protected abstract Class getControllerClass();

    
    private boolean isExtractViewable(ExtractEnablementStatus status) {
    	final boolean viewable = 
    		(ExtractEnablementStatusUtils.ENABLED.equals(status) || 
    		 ExtractEnablementStatusUtils.PARTIAL.equals(status));
    	if (LOG.isDebugEnabled()){
			LOG.debug("Extract " + getExtractType() + " is " 
                + (viewable ? "" : "NOT ") + " viewable");
		}
        return viewable;
    }
    
    private boolean isExtractUpdateable(ExtractEnablementStatus status) {
        final boolean updateable = 
            ExtractEnablementStatusUtils.ENABLED.equals(status);
        if (LOG.isDebugEnabled()){
            LOG.debug("Extract " + getExtractType() + " is " 
                + (updateable ? "" : "NOT ") + " updateable");
        }
        return updateable;
    }
    
    /**
     * Method used to determine if the ExtractEnablement is partial
     * @param e ExtractEnablementStatus for evaluation
     * @return Return true if the Enablement is Partial
     */
    protected boolean hasExtractBeenPartialed(ExtractEnablementStatus e){
		boolean isPartialed = false;
		if(e == ExtractEnablementStatusUtils.PARTIAL){
			isPartialed = true;
		}
		if(LOG.isInfoEnabled()){
			LOG.info("Extract has been marked as partialed-read-only = " 
					+ isPartialed);
		}
    	return isPartialed;
    }
    
    protected void updateExtract(HttpServletRequest request) {
        Precondition.assertNotNull("request", request);
        Patient patient = getPatient(request);
        Precondition.assertNotNull("session.patient", patient);
        ExtractServiceDelegate delegate = ServiceDelegateFactory.
            createExtractServiceDelegate();
        delegate.update(getExtractType(), patient);
    }
    
    private Patient getPatient(HttpServletRequest request) {
        return SessionManagement.getInstance().getPatient(
            request);
    }

    private String getUserName(HttpServletRequest request) {
        Patient patient = SessionManagement.getInstance().getPatient(
            request);
        String userName = null;
        if ((patient != null) && (patient.getUserProfile() != null)) {
            userName = patient.getUserProfile().getUserName();
        }
        return userName;
    }

    private void setPatient(HttpServletRequest request, Patient patient) {
        SessionManagement.getInstance().setPatient(request, patient);
    }

    private boolean isExtractUpdated(HttpServletRequest request) {
        return SessionManagement.getInstance().isExtractUpdated(request, 
            getExtractType());
    }

    private void setExtractUpdated(HttpServletRequest request, 
        boolean updated) 
    {
        SessionManagement.getInstance().setExtractUpdated(request, 
            getExtractType(), updated);
    }

    /**
     * new method to set the enablement status in the modded session management 
     * class.
     * @param request
     * @param status
     */
    private void setExtractEnablementStatus(HttpServletRequest request, 
    		ExtractEnablementStatus status)
    {
    	SessionManagement.getInstance().setExtractEnablementStatus(request,
    			getExtractType(), status);
    }

    /**
     * 
     */
    private String createPortletBeginUrl(HttpServletRequest request, 
        HttpServletResponse response)
    {
       PostbackURL url = PostbackURL.createPostbackURL(request, response);
       String actionParameter = createActionParameter(request, 
           "actionOverride"); 
       url.removeParameter(actionParameter);
       url.addParameter(actionParameter, createPortletActionValue(request,
           "begin"));
       return response.encodeRedirectURL(url.toString());
    }

    private void yieldOnError(ServiceResponse response, String message) { 
        if (response.getMessages().hasErrorMessages()) {
            LOG.error(response.getMessages().getErrorMessages());
            MessagesStringBuilder builder = new MessagesStringBuilder();
            builder.append(response.getMessages(), Patient.class);
            LOG.error(builder.getErrorString());
            String infoString = builder.getInfoString();
            if (!StringUtils.isBlank(infoString)) {
                LOG.info(infoString);
            }
            throw new IllegalStateException(message);
        }
    }

    private InPersonAuthenticationServiceDelegate getIPADelegate() {
        return IPA_DELEGATE;
    }

    private ExtractServiceDelegate getExtractDelegate() {
        return EXTRACT_DELEGATE;
    }

}
